#!/usr/bin/python
# -*- coding: utf-8 -*-

#
#  Media Center Master Plex Agent (movies and T.V.)
#
#     Provides Plex with local movie and T.V. metadata as generated by the
#     Media Center Master standalone client (or any other movie.xml/series.xml-
#     compatible metadata).
#
#     Supports multiple artwork files, season correction, show matching by
#     series IDs in local metadata, DVD/BluRay/HD DVD, and just about every
#     metadata field that Plex provides.
#
#  All original code by copyrighted by Media Center Master:
#      (C) 2013-2018 Media Center Master, Inc.
#      All rights reserved.
#      http://www.MediaCenterMaster.com/
#
#  Version 1.00     2013-02-11
#  Version 1.01     2013-07-16     version consistency with
#                                  scanner (no code change)
#  Version 1.2      2015-02-01     compatible with non-Windows OS
#  Version 1.3      2018-01-27     formatted for PEP8 compliance
#                                  http://www.pep8online.com/
#  Version 1.4      2018-02-13     fixed movie writers list
#                                  fixed movie director
#                                  fixed movie sort title
#                                  support for movie actors
#                                  support for movie collections (sets)
#                                  support for similar movies
#                                  support for movie tags/keywords
#                                  BETA support for theatrical trailers/extras
# Version 1.4.1     2018-02-14     support for movie actor images
#                                  full support for theatrical trailers/extras
# Version 1.5       2018-02-15     fixed displaying multiple movie writers
#                                  fixed T.V. actors
#                                  fixed T.V. writers
#                                  support for movie banner images
#                                  support for T.V. theme music
#                                  improved theatrical trailers/extras support
#                                  improved error handling
# Version 1.6      2018-02-18      version consistency with scanner
#                                  (no code changes)
# Version 1.7      2025-11-17      unicode-safe media file paths
#                                  local trailers via Proxy.LocalFile
#                                  episode thumb path normalization
#                                  durations from VideoLengthSeconds
#                                  voted score/audience rating mapping
#                                  cast filtered to actors only
#

import sys
import os
import re
import hashlib
import unicodedata
import urllib
import urlparse

ALLOW_FALLBACK_PREF = 'allow_fallback'

try:
    STRING_TYPES = basestring
except NameError:
    STRING_TYPES = str


def EnsureUnicodePath(path):
    if path is None:
        return u''
    if isinstance(path, unicode):
        return unicodedata.normalize('NFKC', path)
    fs_encoding = sys.getfilesystemencoding() or 'utf-8'
    try:
        decoded = unicode(path, fs_encoding, 'replace')
    except (TypeError, UnicodeDecodeError):
        decoded = unicode(str(path), 'utf-8', 'replace')
    return unicodedata.normalize('NFKC', decoded)

def allow_fallback_enabled():
    try:
        value = Prefs[ALLOW_FALLBACK_PREF]
    except Exception:
        return False
    if isinstance(value, STRING_TYPES):
        return value.lower() in ('true', '1', 'yes', 'on')
    return bool(value)

def is_youtube_url(url):
    try:
        parsed = urlparse.urlparse(url)
        host = (parsed.netloc or '').lower()
        return host.endswith('youtube.com') or host.endswith('youtu.be') or host.endswith('y2u.be')
    except:
        return False


def Start():
    Log('Media Center Master agent loaded; fallback preference = %s' %
        allow_fallback_enabled())


class MediaCenterMaster(Agent.Movies):

    name = 'Media Center Master'
    primary_provider = True
    languages = [Locale.Language.English]

    def search(self, results, media, lang, manual=False):
        try:
            Log('Identifying "' + media.title + '"...')
            results.Append(MetadataSearchResult(
                id=media.items[0].parts[0].hash,
                year=media.year, name=media.title,
                lang=lang, score=100))
            Log('      ... done')
        except Exception as e:
            self.debugException(e, 'FAILURE/error')
            if allow_fallback:
                Log('Allowing Plex fallback metadata providers (movie exception).')
                return
            raise
        return

    def update(self, metadata, media, lang, force=False):
        allow_fallback = allow_fallback_enabled()
        VideoFilename = ''

        try:
            try:
                VideoFilename = EnsureUnicodePath(
                    media.items[0].parts[0].file)
            except Exception as e:
                self.debugException(e, 'Couldn\'t parse filename.')
                if allow_fallback:
                    Log('Allowing Plex fallback metadata providers (movie path).')
                    return
                raise

            Log('Scraping XML metadata for: ' + VideoFilename)

            MoviePath = os.path.dirname(VideoFilename)

            if self.discSubstructure(MoviePath):
                MoviePath = os.path.abspath(os.path.join(MoviePath, os.pardir))

            XMLFilename = os.path.join(MoviePath, 'movie.xml')

            if not os.path.exists(XMLFilename):
                Log(
                    '      metadata missing, skipping (fetch with Media ' +
                    'Center Master)')
                if allow_fallback:
                    Log('Allowing Plex fallback metadata providers (movie XML missing).')
                    return
                raise Exception('movie.xml missing and fallback disabled.')

            XMLData = XML.ElementFromString(Core.storage.load(XMLFilename))

            Log(
                '      ' + self.getTextFromXML(XMLData, 'LocalTitle') +
                ' (' + self.getTextFromXML(XMLData, 'ProductionYear') + ')')

            metadata.title = self.getTextFromXML(XMLData, 'LocalTitle')
            metadata.original_title = self.getTextFromXML(
                XMLData, 'OriginalTitle')
            try:
                metadata.title_sort = self.getTextFromXML(
                    XMLData, 'SortTitle')
            except:
                pass
            metadata.summary = self.getTextFromXML(XMLData, 'Description')
            metadata.tagline = self.getTextFromXML(XMLData, 'Tagline')
            metadata.content_rating = self.getTextFromXML(
                XMLData, 'MPAARating')
            # Plex only holds one studio
            metadata.studio = self.getTextFromXML(
                XMLData, 'Studios/Studio'
            )
            try:
                metadata.year = int(
                    self.getTextFromXML(XMLData, 'ProductionYear'))
            except:
                pass
            try:
                imdb_rating_value = self.getTextFromXML(
                    XMLData, 'IMDBrating')
                if imdb_rating_value:
                    imdb_rating = float(imdb_rating_value)
                    metadata.rating = imdb_rating
                    metadata.audience_rating = imdb_rating
            except:
                pass
            video_length_seconds = self.getTextFromXML(
                XMLData, 'VideoLengthSeconds')
            if video_length_seconds:
                try:
                    metadata.duration = int(video_length_seconds) * 1000
                except:
                    pass
            else:
                try:
                    # convert minutes to milliseconds
                    metadata.duration = int(
                        self.getTextFromXML(XMLData, 'RunningTime')) * 1000 * 60
                except:
                    pass

            try:
                temp_country = self.getTextFromXML(XMLData, 'Country')
                if temp_country not in metadata.countries:
                    metadata.countries.add(temp_country)
            except:
                pass

            try:
                actors_xml = XMLData.xpath('//Title/Persons/Person')
                metadata.roles.clear()
                for this_actor in actors_xml:
                    person_name = this_actor[0].text
                    person_role = this_actor[2].text or ''
                    person_type = ''
                    try:
                        person_type = this_actor[1].text or ''
                    except:
                        person_type = ''
                    if person_role != '' and \
                            (person_type == '' or
                             person_type.lower() == 'actor'):
                        new_actor_data = metadata.roles.new()
                        new_actor_data.name = person_name
                        try:
                            if this_actor[3].text != '':
                                new_actor_data.photo = this_actor[3].text
                        except:
                            pass
                        new_actor_data.role = person_role
            except Exception as e:
                self.debugException(e, 'Failure to parse actors XML')
                pass

            try:
                metadata.rating_count = \
                    int(self.getTextFromXML(XMLData, 'Votes'))
            except Exception as e:
                self.debugException(e, 'Failure to parse rating count XML')
                pass

            try:
                temp_director = self.getTextFromXML(XMLData, 'Director')
                metadata.directors.clear()
                new_director_data = metadata.directors.new()
                new_director_data.name = temp_director
            except:
                pass

            try:
                temp_writers = self.getTextFromXML(
                    XMLData, 'WritersList').split('|')
                metadata.writers.clear()
                for w in temp_writers:
                    new_writer = metadata.writers.new()
                    new_writer.name = w
            except:
                pass

            try:
                similar_movies_xml = XMLData.xpath(
                    '//Title/SimilarMovies/SimilarMovie')
                metadata.similar.clear()
                for this_similar_movie in similar_movies_xml:
                    metadata.similar.add(this_similar_movie.text)
            except:
                pass

            try:
                temp_genres = self.getTagsFromXML(
                    XMLData, 'Genres/Genre')
                metadata.genres.clear()
                for g in temp_genres:
                    if g not in metadata.genres:
                        metadata.genres.add(g.text)
            except:
                pass

            try:
                temp_tags = self.getTagsFromXML(
                    XMLData, 'Tags/Tag')
                metadata.tags.clear()
                for g in temp_tags:
                    if g not in metadata.tags:
                        metadata.tags.add(g.text)
            except:
                pass

            try:
                temp_collection = self.getTextFromXML(
                    XMLData, 'CollectionName')
                metadata.collections.clear()
                if temp_collection != '':
                    metadata.collections.add(temp_collection)
            except Exception as e:
                self.debugException(e, 'Failure to collections XML')
                pass

            try:
                TrailerFileToUse = ''
                DiscoveredTrailerFile = self.discoverLocalTrailerFile(
                    VideoFilename)
                XMLLocalTrailerFile = self.getTextFromXML(
                    XMLData, 'LocalTrailer/URL')
                XMLOnlineTrailerURL = self.getTextFromXML(
                    XMLData, 'TrailerURL')

                if DiscoveredTrailerFile != '':
                    TrailerFileToUse = DiscoveredTrailerFile
                    Log('      ... trailer file (discovered): ' +
                        DiscoveredTrailerFile)
                if XMLLocalTrailerFile != '':
                    Log('      ... trailer file (from metadata): ' +
                        XMLLocalTrailerFile)
                    if not os.path.exists(XMLLocalTrailerFile):
                        XMLLocalTrailerFile = ''
                        Log('         ... file does not exist!')
                if TrailerFileToUse == '':
                    if XMLLocalTrailerFile != '':
                        TrailerFileToUse = XMLLocalTrailerFile
                if XMLOnlineTrailerURL != '':
                    Log('      ... trailer URL (from metadata): ' +
                        XMLOnlineTrailerURL)

                if TrailerFileToUse != '':
                    avail = Datetime.ParseDate(self.getTextFromXML(
                        XMLData, 'ReleaseDate'))
                    if os.path.exists(TrailerFileToUse):
                        metadata.extras.add(TrailerObject(
                            file=Proxy.LocalFile(TrailerFileToUse),
                            title=metadata.title + ' trailer',
                            year=metadata.year,
                            originally_available_at=avail,
                            thumb=''))
                    else:
                        Log('      ... trailer missing at playback: ' +
                            TrailerFileToUse)
                elif XMLOnlineTrailerURL != '':
                    # Note: this is currently broken for YouTube
                    # trailers!  See:  https://github.com/plexinc-plugins/
                    #                      Services.bundle/issues/943

                    if is_youtube_url(XMLOnlineTrailerURL):
                        Log('      ... skipped unsupported YouTube trailer URL: ' +
                            XMLOnlineTrailerURL)
                    else:
                        avail = Datetime.ParseDate(self.getTextFromXML(
                            XMLData, 'ReleaseDate'))
                        metadata.extras.add(TrailerObject(
                            url=XMLOnlineTrailerURL,
                            title=metadata.title + ' trailer',
                            year=metadata.year,
                            originally_available_at=avail,
                            thumb=''))

            except Exception as e:
                self.debugException(e, 'Failure to parse trailer XML')
                pass

            try:
                metadata.originally_available_at = Datetime.ParseDate(
                    self.getTextFromXML(
                        XMLData, 'ReleaseDate')).date()
            except Exception as e:
                self.debugException(e, 'Failure to date released XML')
                pass

            try:
                if os.path.exists(
                    os.path.join(MoviePath, 'folder.jpg')
                ):
                    poster_data = Core.storage.load(
                        os.path.join(MoviePath, 'folder.jpg'))
                    poster_name = hashlib.md5(poster_data).hexdigest()
                    if poster_name not in metadata.posters:
                        metadata.posters[poster_name] = \
                            Proxy.Media(poster_data)
            except:
                pass

            try:
                for i in range(0, 99):
                    sBackdropName = ''
                    if i == 0:
                        sBackdropName = os.path.join(
                            MoviePath, 'backdrop.jpg')
                    else:
                        sBackdropName = os.path.join(
                            MoviePath, 'backdrop' + str(i) + '.jpg')
                    if os.path.exists(sBackdropName):
                        art_data = Core.storage.load(sBackdropName)
                        art_name = hashlib.md5(art_data).hexdigest()
                        if art_name not in metadata.art:
                            metadata.art[art_name] = \
                                Proxy.Media(art_data)
            except:
                pass

            try:
                if os.path.exists(
                    os.path.join(MoviePath, 'banner.jpg')
                ):
                    banner_data = Core.storage.load(
                        os.path.join(MoviePath, 'banner.jpg'))
                    banner_name = hashlib.md5(
                        banner_data).hexdigest()
                    if banner_name not in metadata.banners:
                        metadata.banners[banner_name] = \
                            Proxy.Media(banner_data)
            except:
                pass

        except Exception as e:
            self.debugException(e, 'FAILURE/error')
            pass

        Log('      ... done!')
        return

    def debugException(self, exc, desc):
        Log(desc)
        Log('      *** ' + str(type(exc)) + ' ***')
        Log('      *** ' + str(exc) + ' ***')
        exc_type, exc_obj, exc_tb = sys.exc_info()
        fname = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
        lno = exc_tb.tb_lineno - 42  # Plex adds 42 lines of code
        Log('      *** ' + fname + ':' + str(lno) + ' ***')
        return

    def discoverLocalTrailerFile(self, VideoFilename):
        MoviePath = os.path.dirname(VideoFilename)
        TrailerFile = ''

        try:
            MovieRawFile = os.path.basename(VideoFilename)
            (MovieRawFileNoExt, MovieRawFileExt) = \
                os.path.splitext(MovieRawFile)

            TrailerFileTest = os.path.join(
                MoviePath,
                'trailer.mp4')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    'trailer.flv')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    'trailer.avi')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    'trailer.mov')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    'trailer.mkv')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    MovieRawFileNoExt + '-trailer.mp4')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    MovieRawFileNoExt + '-trailer.flv')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    MovieRawFileNoExt + '-trailer.avi')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
            else:
                TrailerFileTest = os.path.join(
                    MoviePath,
                    MovieRawFileNoExt + '-trailer.mov')

            if os.path.exists(TrailerFileTest):
                TrailerFile = TrailerFileTest
        except:
            pass

        return TrailerFile

    def path2url(self, path):
        return urlparse.urljoin(
            'file:', urllib.pathname2url(path))

    def getTagsFromXML(self, XMLData, tag):
        try:
            return XMLData.xpath('//Title/' + tag)
        except:
            pass
        return {}

    def getTextFromXML(self, XMLData, tag):
        try:
            return self.getTagsFromXML(XMLData, tag)[0].text
        except:
            pass
        return ''

    def discSubstructure(self, fullPath):
        fullPath = fullPath.upper().replace('/', '\\')
        if fullPath.endswith('\\VIDEO_TS'):
            return True
        if fullPath.endswith('\\BDMV'):
            return True
        if fullPath.endswith('\\HVDVD_TS'):
            return True
        return False


class MediaCenterMaster(Agent.TV_Shows):

    name = 'Media Center Master'
    primary_provider = True
    languages = [Locale.Language.English]

    def search(self, results, media, lang, manual=False):

        try:
            Log('Identifying "' + media.title + '"...')

            results.Append(
                MetadataSearchResult(
                    id='MCM_TV_X_' + str(media.id),
                    year=media.year, name=media.title, lang=lang, score=50))

            try:

                # Let's try to get a thetvdb.com, tvrage.com, or imdb.com ID
                # to uniquely identify # this series (helps prevent series
                # from breaking up so much).

                EpisodeFilename = ''

                try:
                    # Python v3
                    EpisodeFilename = urllib.parse.unquote(media.filename)
                except:
                    # Python v2
                    EpisodeFilename = urllib.unquote(media.filename)

                ShowPath = os.path.abspath(
                    os.path.join(os.path.dirname(EpisodeFilename), os.pardir))
                ShowMetadata = os.path.join(ShowPath, 'series.xml')
                ShowXML = \
                    XML.ElementFromString(Core.storage.load(ShowMetadata))
                iSeriesID = int(self.getTextFromShowXML(ShowXML, 'id'))

                if iSeriesID > 0:
                    results.Append(
                        MetadataSearchResult(
                            id='MCM_TV_A_' + str(iSeriesID), year=media.year,
                            name=media.title, lang=lang, score=100))
                else:
                    iSeriesID = int(self.getTextFromShowXML(ShowXML,
                                    'TVRageID'))

                    if iSeriesID > 0:
                        results.Append(
                            MetadataSearchResult(
                                id='MCM_TV_B_' + str(iSeriesID),
                                year=media.year, name=media.title, lang=lang,
                                score=100))
                    else:
                        sSeriesID = self.getTextFromShowXML(ShowXML, 'IMDbId')

                        if len(sSeriesID) > 8:
                            results.Append(
                                MetadataSearchResult(
                                    id='MCM_TV_C_' + sSeriesID,
                                    year=media.year, name=media.title,
                                    lang=lang, score=100))
            except:

                pass

            Log('      ... done')
        except Exception as e:
            self.debugException(e, 'FAILURE/error')
            pass
        return

    def update(self, metadata, media, lang, force=False):
        allow_fallback = allow_fallback_enabled()
        try:
            bHasShowMeta = False
            ShowXML = None

            for s in media.seasons:

                # Log('Searching through season: %s', s)

                # Sanity check, this doesn't really do much
                metadata.seasons[s].index = int(s)
                SeasonPath = ''

                try:
                    FirstEpisodeFile = \
                        EnsureUnicodePath(
                            media.seasons[s].episodes[e].items[0].parts[0].file)
                    SeasonPath = os.path.dirname(FirstEpisodeFile)

                    if os.path.exists(os.path.join(SeasonPath, 'folder.jpg')):
                        poster_data = Core.storage.load(
                            os.path.join(SeasonPath, 'folder.jpg'))
                        poster_name = \
                            hashlib.md5(poster_data).hexdigest()
                        if poster_name not in metadata.seasons[s].posters:
                            metadata.seasons[s].posters[poster_name] = \
                                Proxy.Media(poster_data)
                except:
                    pass

                for e in media.seasons[s].episodes:
                    episode = metadata.seasons[s].episodes[e]

                    try:
                        EpisodeFile = \
                            EnsureUnicodePath(
                                media.seasons[s].episodes[e].items[0].parts[0].file)
                        SeasonPath = os.path.dirname(EpisodeFile)

                        if bHasShowMeta is False:
                            ShowPath = os.path.abspath(
                                os.path.join(
                                    os.path.dirname(EpisodeFile), os.pardir))
                            ShowMetadata = os.path.join(ShowPath, 'series.xml')
                            Log('    show metadata: ' + ShowMetadata)
                            ShowXML = XML.ElementFromString(
                                Core.storage.load(ShowMetadata))
                            bHasShowMeta = True
                            metadata.title = \
                                self.getTextFromShowXML(ShowXML, 'SeriesName')
                            metadata.summary = \
                                self.getTextFromShowXML(ShowXML, 'Overview')
                            try:  # convert minutes to milliseconds
                                metadata.duration = int(
                                    self.getTextFromShowXML(
                                        ShowXML, 'Runtime')) * 1000 * 60
                            except:
                                pass

                            try:
                                temp_actors = self.getTextFromShowXML(
                                    ShowXML, 'Actors').split('|')
                                metadata.roles.clear()
                                for this_actor in temp_actors:
                                    if this_actor != '':
                                        new_actor_data = metadata.roles.new()
                                        new_actor_data.name = this_actor
                                        # Show actors list contains actors only,
                                        # so no additional filtering needed.
                            except Exception as e:
                                self.debugException(
                                    e, 'Failure to parse actors XML')
                                pass

                            metadata.studio = \
                                self.getTextFromShowXML(ShowXML, 'Network')
                            metadata.content_rating = self.getTextFromShowXML(
                                ShowXML, 'ContentRating')
                            try:
                                metadata.rating = float(
                                    self.getTextFromShowXML(ShowXML, 'Rating')
                                )
                            except:
                                pass

                            try:
                                imdb_rating_show = self.getTextFromShowXML(
                                    ShowXML, 'IMDBrating')
                                if imdb_rating_show:
                                    metadata.audience_rating = float(
                                        imdb_rating_show)
                            except:
                                pass

                            try:
                                metadata.originally_available_at = \
                                    Datetime.ParseDate(
                                        self.getTextFromShowXML(
                                            ShowXML, 'FirstAired')).date()
                            except:
                                pass

                            try:
                                temp_genres = self.getTextFromShowXML(
                                    ShowXML, 'Genre').split('|')
                                metadata.genres.clear()
                                for g in temp_genres:
                                    if len(g) > 0:
                                        if g not in metadata.genres:
                                            metadata.genres.add(g)
                            except:
                                pass

                            try:
                                if os.path.exists(
                                    os.path.join(ShowPath, 'folder.jpg')
                                ):
                                    poster_data = Core.storage.load(
                                        os.path.join(ShowPath, 'folder.jpg'))
                                    poster_name = hashlib.md5(
                                        poster_data).hexdigest()
                                    if poster_name not in metadata.posters:
                                        metadata.posters[poster_name] = \
                                            Proxy.Media(poster_data)
                            except:
                                pass

                            try:
                                if os.path.exists(
                                    os.path.join(ShowPath, 'banner.jpg')
                                ):
                                    banner_data = Core.storage.load(
                                        os.path.join(ShowPath, 'banner.jpg'))
                                    banner_name = hashlib.md5(
                                        banner_data).hexdigest()
                                    if banner_name not in metadata.banners:
                                        metadata.banners[banner_name] = \
                                            Proxy.Media(banner_data)
                            except:
                                pass

                            try:
                                for i in range(0, 99):
                                    sBackdropName = ''
                                    if i == 0:
                                        sBackdropName = os.path.join(
                                            ShowPath, 'backdrop.jpg')
                                    else:
                                        sBackdropName = os.path.join(
                                            ShowPath,
                                            'backdrop' + str(i) + '.jpg')
                                    if os.path.exists(sBackdropName):
                                        backdrop_data = Core.storage.load(
                                            sBackdropName)
                                        backdrop_name = hashlib.md5(
                                            backdrop_data).hexdigest()
                                        if backdrop_name not in metadata.art:
                                            metadata.art[backdrop_name] = \
                                                Proxy.Media(backdrop_data)
                            except:
                                pass

                            try:
                                if os.path.exists(
                                    os.path.join(ShowPath, 'theme.mp3')
                                ):
                                    theme_data = Core.storage.load(
                                        os.path.join(ShowPath, 'theme.mp3'))
                                    theme_name = hashlib.md5(
                                        theme_data).hexdigest()
                                    if theme_name not in metadata.themes:
                                        metadata.themes[theme_name] = \
                                            Proxy.Media(theme_data)
                            except:
                                pass

                        Log('    video: ' + EpisodeFile)

                        (EpisodeFileNoExt, EpisodeFileExt) = \
                            os.path.splitext(EpisodeFile)
                        EpisodeMetadata = os.path.join(
                            os.path.dirname(EpisodeFile),
                            'metadata',
                            os.path.basename(EpisodeFileNoExt) + '.xml')
                        Log('        metadata: ' + EpisodeMetadata)

                        if not os.path.exists(EpisodeMetadata):
                            Log(
                                '            metadata file missing, ' +
                                'skipping (fetch with Media Center Master)')
                            continue

                        EpisodeXML = XML.ElementFromString(
                            Core.storage.load(EpisodeMetadata))

                        Log(
                            '        episode: ' +
                            self.getTextFromEpisodeXML(
                                EpisodeXML, 'EpisodeName'))

                        # Try to correct the season number

                        try:
                            MCMSeason = -1

                            sSeasonNumber = self.getTextFromEpisodeXML(
                                EpisodeXML, 'SeasonNumber')

                            if sSeasonNumber != '':
                                MCMSeason = int(sSeasonNumber)
                            else:
                                SeasonFolderOnly = os.path.basename(
                                    os.path.normpath(SeasonPath))
                                MCMSeason = -1
                                match = re.match(
                                    '.*?(?P<season>[0-9]+)$',
                                    SeasonFolderOnly,
                                    re.IGNORECASE)
                                if match:
                                    MCMSeason = int(match.group('season'))

                            if MCMSeason > -1:
                                metadata.seasons[s].index = MCMSeason
                                # this should exist, per http://dev.plexapp.com
                                # /docs/api/objectkit.html#EpisodeObject
                                try:
                                    episode.season = MCMSeason
                                except:
                                    pass
                        except:

                            pass

                        episode.title = self.getTextFromEpisodeXML(
                            EpisodeXML, 'EpisodeName')
                        episode.summary = self.getTextFromEpisodeXML(
                            EpisodeXML, 'Overview')

                        video_length_seconds = self.getTextFromEpisodeXML(
                            EpisodeXML, 'VideoLengthSeconds')
                        if video_length_seconds:
                            try:
                                episode.duration = \
                                    int(video_length_seconds) * 1000
                            except:
                                pass
                        # Else keep Plex auto duration behavior

                        try:
                            episode.rating = float(
                                self.getTextFromEpisodeXML(
                                    EpisodeXML, 'Rating'))
                        except:
                            pass

                        try:
                            episode.absolute_index = int(
                                self.getTextFromEpisodeXML(
                                    EpisodeXML, 'absolute_number'))
                        except:
                            pass

                        try:
                            episode.originally_available_at = \
                                Datetime.ParseDate(
                                    self.getTextFromEpisodeXML(
                                        EpisodeXML, 'FirstAired')).date()
                        except:
                            pass

                        try:
                            temp_writers = self.getTextFromEpisodeXML(
                                EpisodeXML, 'Writer').split(', ')
                            episode.writers.clear()
                            for w in temp_writers:
                                new_writer = episode.writers.new()
                                new_writer.name = w
                        except:
                            pass

                        try:
                            episode.directors.clear()
                            episode.directors.add(
                                self.getTextFromEpisodeXML(
                                    EpisodeXML, 'Director'))
                        except:
                            pass

                        try:
                            temp_guests = self.getTextFromEpisodeXML(
                                EpisodeXML, 'GuestStars').split('|')
                            episode.guest_stars.clear()
                            for g in temp_guests:
                                if g != '':
                                    new_guest_star = episode.guest_stars.new()
                                    new_guest_star.name = g
                        except:
                            pass

                        try:
                            thumb_name_xml = self.getTextFromEpisodeXML(
                                EpisodeXML, 'filename')
                            if thumb_name_xml:
                                normalized_thumb = thumb_name_xml.lstrip('\\/')
                                thumb_file = os.path.join(
                                    os.path.dirname(EpisodeFile),
                                    'metadata',
                                    normalized_thumb)
                                if thumb_file.upper().endswith('.JPG'):
                                    if os.path.exists(thumb_file):
                                        thumb_data = Core.storage.load(
                                            thumb_file)
                                        thumb_name = hashlib.md5(
                                            thumb_data).hexdigest()
                                        if thumb_name not in episode.thumbs:
                                            episode.thumbs[thumb_name] = \
                                                Proxy.Media(thumb_data)
                        except:
                            pass
                    except Exception as e:
                        self.debugException(e, 'EPISODEFAILURE/error')
                        pass
        except Exception as e:
            self.debugException(e, 'FAILURE/error')
            if allow_fallback:
                Log('Allowing Plex fallback metadata providers (TV exception).')
                return
            raise

        Log('    ... done!')
        return

    def getTextFromShowXML(self, XMLData, tag):
        try:
            return XMLData.xpath('//Series/' + tag)[0].text
        except:
            pass
        return ''

    def getTextFromEpisodeXML(self, XMLData, tag):
        try:
            return XMLData.xpath('//Item/' + tag)[0].text
        except:
            pass
        return ''

    def debugException(self, exc, desc):
        Log(desc)
        Log('      *** ' + str(type(exc)) + ' ***')
        Log('      *** ' + str(exc) + ' ***')
        exc_type, exc_obj, exc_tb = sys.exc_info()
        fname = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
        lno = exc_tb.tb_lineno - 42  # Plex adds 42 lines of code
        Log('      *** ' + fname + ':' + str(lno) + ' ***')
        return
